// modules/maps/frontend/route-handler.js
// Клиентская логика обработки маршрута и аудиогида

(async function() {
  // Порог срабатывания в метрах
  const THRESHOLD = 50;

  // Загружаем список POI из JSON
  const res = await fetch('/modules/maps/config/route.json');
  const poiList = await res.json();

  // Восстанавливаем те POI, которые уже были проиграны
  const played = new Set(JSON.parse(localStorage.getItem('playedPOIs') || '[]'));

  // Функция вычисления расстояния между двумя координатами (Haversine)
  function getDistance(lat1, lon1, lat2, lon2) {
    const R = 6371000; // радиус Земли в метрах
    const dLat = (lat2 - lat1) * Math.PI / 180;
    const dLon = (lon2 - lon1) * Math.PI / 180;
    const a = Math.sin(dLat/2) * Math.sin(dLat/2) +
              Math.cos(lat1 * Math.PI/180) * Math.cos(lat2 * Math.PI/180) *
              Math.sin(dLon/2) * Math.sin(dLon/2);
    const c = 2 * Math.atan2(Math.sqrt(a), Math.sqrt(1-a));
    return R * c;
  }

  // Воспроизведение аудио для POI
  function playAudio(poi) {
    const audio = new Audio(poi.audio_url);
    audio.play().catch(err => console.error('Ошибка при воспроизведении аудио:', err));
    played.add(poi.id);
    localStorage.setItem('playedPOIs', JSON.stringify([...played]));
    // TODO: можно изменить иконку маркера или отрисовать уведомление
  }

  // Проверяем приближение к POI и запускаем аудио
  function checkProximity(position) {
    const { latitude, longitude } = position.coords;
    poiList.forEach(poi => {
      if (!played.has(poi.id)) {
        const dist = getDistance(latitude, longitude, poi.lat, poi.lng);
        if (dist <= THRESHOLD) {
          playAudio(poi);
        }
      }
    });
  }

  // Запускаем отслеживание позиции
  if (navigator.geolocation) {
    navigator.geolocation.watchPosition(
      checkProximity,
      err => console.error('Ошибка геолокации:', err),
      { enableHighAccuracy: true, maximumAge: 0, timeout: 10000 }
    );
  } else {
    console.warn('Геолокация не поддерживается в этом браузере');
  }
})();
