// modules/excursions/backend/js/osm_tourism_overlay.js
(function(){
  // Значок для OSM tourism POI
  const tourismIcon = L.icon({
    iconUrl: 'https://cdn-icons-png.flaticon.com/512/684/684908.png',
    iconSize: [24, 24],
    iconAnchor: [12, 12]
  });

  /**
   * Добавляет слой с точками туризма один раз при инициализации
   * @param {L.Map} map — экземпляр Leaflet-карты
   * @returns {L.LayerGroup}
   */
  function addTourismLayer(map) {
    const overlay = L.layerGroup().addTo(map);

    async function fetchPois() {
      try {
        const b = map.getBounds();
        const bbox = [b.getSouth(), b.getWest(), b.getNorth(), b.getEast()].join(',');
        const query =
          `[out:json][timeout:15];` +
          `node["tourism"](${bbox});` +
          `out center;`;

        const res = await fetch('https://overpass-api.de/api/interpreter', {
          method: 'POST',
          body: query
        });
        if (!res.ok) {
          console.warn('Overpass API error:', res.status);
          return;
        }
        const data = await res.json();
        overlay.clearLayers();
        (data.elements || []).forEach(el => {
          const lat = el.lat ?? el.center?.lat;
          const lon = el.lon ?? el.center?.lon;
          if (!lat || !lon) return;
          const name = el.tags?.name || el.tags?.tourism || 'POI';
          L.marker([lat, lon], { icon: tourismIcon })
            .bindPopup(`<strong>${name}</strong>`)
            .addTo(overlay);
        });
      } catch (err) {
        console.error('Ошибка загрузки OSM-POI:', err);
      }
    }

    // Загружаем лишь единожды
    fetchPois();

    // Если хотите обновлять вручную — можно раскомментировать кнопку:
    /*
    const btn = L.control({ position: 'topright' });
    btn.onAdd = () => {
      const d = L.DomUtil.create('div','refresh-poi');
      d.innerHTML = '<button title="Обновить POI">🔄</button>';
      d.firstChild.onclick = fetchPois;
      return d;
    };
    btn.addTo(map);
    */

    return overlay;
  }

  // Экспорт в глобальную область
  window.addTourismLayer = addTourismLayer;
})();
